// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef GPU_COMMAND_BUFFER_SERVICE_DECODER_CONTEXT_H_
#define GPU_COMMAND_BUFFER_SERVICE_DECODER_CONTEXT_H_

#include <stddef.h>
#include <stdint.h>

#include <string>

#include "base/memory/scoped_refptr.h"
#include "base/memory/weak_ptr.h"
#include "gpu/command_buffer/common/capabilities.h"
#include "gpu/command_buffer/common/constants.h"
#include "gpu/command_buffer/common/context_result.h"
#include "gpu/command_buffer/service/async_api_interface.h"
#include "gpu/gpu_gles2_export.h"

namespace gl {
class GLContext;
class GLImage;
class GLSurface;
}  // namespace gl

namespace gpu {
class TextureBase;
struct ContextCreationAttribs;

namespace gles2 {
class ContextGroup;
class ErrorState;
class GpuFenceManager;
class QueryManager;
struct ContextState;
struct DisallowedFeatures;
}  // namespace gles2

// Abstract interface implemented by {Raster,GLES2}Decoder. It is called a
// DecoderContext because all methods are about decoding commands or
// accessing context/state generated by decoded commands.
class GPU_GLES2_EXPORT DecoderContext : public AsyncAPIInterface {
 public:
  DecoderContext() = default;
  ~DecoderContext() override = default;

  //
  // Methods required by CommandBufferStub.
  //
  // Initializes the graphics context. Can create an offscreen
  // decoder with a frame buffer that can be referenced from the parent.
  // Takes ownership of GLContext.
  // Parameters:
  //  surface: the GL surface to render to.
  //  context: the GL context to render to.
  //  offscreen: whether to make the context offscreen or not. When FBO 0 is
  //      bound, offscreen contexts render to an internal buffer, onscreen ones
  //      to the surface.
  //  offscreen_size: the size if the GL context is offscreen.
  // Returns:
  //   true if successful.
  virtual gpu::ContextResult Initialize(
      const scoped_refptr<gl::GLSurface>& surface,
      const scoped_refptr<gl::GLContext>& context,
      bool offscreen,
      const gles2::DisallowedFeatures& disallowed_features,
      const ContextCreationAttribs& attrib_helper) = 0;

  // Destroys the graphics context.
  virtual void Destroy(bool have_context) = 0;

  virtual Capabilities GetCapabilities() = 0;

  // Gets the associated GLContext.
  virtual gl::GLContext* GetGLContext() = 0;

  // Make this decoder's GL context current.
  virtual bool MakeCurrent() = 0;

  // Lose this context.
  virtual void MarkContextLost(error::ContextLostReason reason) = 0;

  // Returns true if the context was lost either by GL_ARB_robustness, forced
  // context loss or command buffer parse error.
  virtual bool WasContextLost() const = 0;

  // Returns true if the context was lost specifically by GL_ARB_robustness.
  virtual bool WasContextLostByRobustnessExtension() const = 0;

  // Updates context lost state and returns true if lost. Most callers can use
  // WasContextLost() as the GLES2Decoder will update the state internally. But
  // if making GL calls directly, to the context then this state would not be
  // updated and the caller can use this to determine if their calls failed due
  // to context loss.
  virtual bool CheckResetStatus() = 0;

  // Gets the QueryManager for this context.
  virtual gles2::QueryManager* GetQueryManager() = 0;

  // Gets the GpuFenceManager for this context.
  virtual gles2::GpuFenceManager* GetGpuFenceManager() = 0;

  // Returns false if there are no pending queries.
  virtual bool HasPendingQueries() const = 0;

  // Process any pending queries.
  virtual void ProcessPendingQueries(bool did_finish) = 0;

  // Returns false if there is no idle work to be made.
  virtual bool HasMoreIdleWork() const = 0;

  // Perform any idle work that needs to be made.
  virtual void PerformIdleWork() = 0;

  // Whether there is state that needs to be regularly polled.
  virtual bool HasPollingWork() const = 0;

  // Perform necessary polling.
  virtual void PerformPollingWork() = 0;

  //
  // Methods required by GLStateRestorerImpl.
  //
  virtual bool initialized() const = 0;
  virtual const gles2::ContextState* GetContextState() = 0;

  // Restores all of the decoder GL state.
  virtual void RestoreState(const gles2::ContextState* prev_state) = 0;

  // Restore States.
  virtual void RestoreActiveTexture() const = 0;
  virtual void RestoreAllTextureUnitAndSamplerBindings(
      const gles2::ContextState* prev_state) const = 0;
  virtual void RestoreActiveTextureUnitBinding(unsigned int target) const = 0;
  virtual void RestoreBufferBinding(unsigned int target) = 0;
  virtual void RestoreBufferBindings() const = 0;
  virtual void RestoreFramebufferBindings() const = 0;
  virtual void RestoreRenderbufferBindings() = 0;
  virtual void RestoreProgramBindings() const = 0;
  virtual void RestoreTextureState(unsigned service_id) const = 0;
  virtual void RestoreTextureUnitBindings(unsigned unit) const = 0;
  virtual void RestoreVertexAttribArray(unsigned index) = 0;
  virtual void RestoreAllExternalTextureBindingsIfNeeded() = 0;

  //
  // Methods required by GpuVideoDecodeAccelerator.
  //
  // Gets the texture object associated with the client ID.  null is returned on
  // failure or if the texture has not been bound yet.
  virtual TextureBase* GetTextureBase(uint32_t client_id) = 0;
  virtual void BindImage(uint32_t client_texture_id,
                         uint32_t texture_target,
                         gl::GLImage* image,
                         bool can_bind_to_sampler) = 0;
  virtual base::WeakPtr<DecoderContext> AsWeakPtr() = 0;

  //
  // Methods required by GLES2DecoderHelper.
  //
  virtual gles2::ContextGroup* GetContextGroup() = 0;
  virtual gles2::ErrorState* GetErrorState() = 0;
};

}  // namespace gpu

#endif  // GPU_COMMAND_BUFFER_SERVICE_DECODER_CONTEXT_H_
