#version 330 core

// Declarations
#if USE_DIRECTIONAL_LIGHT || USE_POINT_LIGHT || USE_SPOT_LIGHT
#define NEED_LIGHTING 1
#endif

#if USE_NORMAL_MAP
#define NEED_TANGENT_SPACE 1
#endif

#if USE_SHADOW_MAP || USE_COOKIE_MAP
#define NEED_LIGHT_SPACE 1
#endif

#if USE_DIRECTIONAL_LIGHT
struct Light
{
	vec3 direction;
	vec3 color;
	float intensity;
	float decay;
};
#elif USE_POINT_LIGHT
struct Light
{
	vec3 position;
	vec3 color;
	float intensity;
	float decay;
};
#elif USE_SPOT_LIGHT
struct Light
{
	vec3 position;
	vec3 direction;
	vec3 color;
	float intensity;
	float decay;
	float innerAngleCos;
	float outerAngleCos;
};
#endif

struct Material
{
	vec3 emissive;
	vec3 diffuse;
	vec3 specular;
	float shininess;
	float opacity;
};

// Inputs
in vec3 ex_PositionInView;
#if NEED_LIGHT_SPACE
in vec4 ex_PositionInLight;
#endif
in vec3 ex_NormalInView;
in vec2 ex_TexCoord;
#if NEED_TANGENT_SPACE
in vec3 ex_TangentInView;
in vec3 ex_BitangentInView;
#endif

// Outputs
out vec4 out_FragColor;

// Uniforms
#if USE_DIFFUSE_MAP
uniform sampler2D diffuseSampler;
#endif
#if USE_SPECULAR_MAP
uniform sampler2D specularSampler;
#endif
#if USE_NORMAL_MAP
uniform sampler2D normalSampler;
#endif
#if USE_SHADOW_MAP
uniform sampler2DShadow shadowSampler;
#endif
#if USE_COOKIE_MAP
	#if USE_POINT_LIGHT
		uniform samplerCube cookieSampler;
	#else
		uniform sampler2D cookieSampler;
	#endif
#endif

#if NEED_LIGHTING
uniform Light uLight;
#endif
uniform Material uMaterial;

// Camera parameters
uniform float uNearPlane;
uniform float uFarPlane;

uniform float uRcpShadowSize;

void main()
{
	vec3 diffuseReflectance = uMaterial.diffuse;
	vec3 specularReflectance = uMaterial.specular;

#if !NEED_LIGHTING
	#if USE_DIFFUSE_MAP
		diffuseReflectance *= texture(diffuseSampler, ex_TexCoord).rgb;
	#endif

	out_FragColor = vec4(diffuseReflectance, uMaterial.opacity);
#else
	vec3 lightColor = uLight.color;

	#if USE_DIFFUSE_MAP
		diffuseReflectance *= pow(texture(diffuseSampler, ex_TexCoord).rgb, vec3(2.2));
	#endif

	#if USE_SPECULAR_MAP
		specularReflectance *= pow(texture(specularSampler, ex_TexCoord).rgb, vec3(2.2));
	#endif

	#if USE_POINT_LIGHT || USE_SPOT_LIGHT
		vec3 posToLight = uLight.position - ex_PositionInView;
	#endif

	#if NEED_TANGENT_SPACE
		vec3 T = normalize(ex_TangentInView);
		vec3 B = normalize(ex_BitangentInView);
		vec3 N = normalize(ex_NormalInView);

		mat3 tangentMatrixInverse = mat3(
			T.x, T.y, T.z,
			B.x, B.y, B.z,
			N.x, N.y, N.z
		);

		N = -1.0 + 2.0 * texture(normalSampler, ex_TexCoord).rgb;
		N = normalize(tangentMatrixInverse * N);
	#else
		vec3 N = normalize(ex_NormalInView);
	#endif

	#if USE_DIRECTIONAL_LIGHT
		vec3 L = normalize(-uLight.direction);
	#elif USE_POINT_LIGHT || USE_SPOT_LIGHT
		vec3 L = normalize(posToLight);
	#endif

	// compute attenuation
	#if USE_DIRECTIONAL_LIGHT
		float att = 1.;
	#endif

	#if USE_POINT_LIGHT || USE_SPOT_LIGHT
		float r = length(posToLight);
		float r2 = r * r;
		float d2 = uLight.decay * uLight.decay;
		float att = d2 / (d2 + r2);

		if (r < uLight.decay)
			att = att * (uLight.decay - r) / uLight.decay;
		else
			att = 0.;
	#endif

	#if USE_SHADOW_MAP
		float shadowTerm = 0.;
	#else
		float shadowTerm = 1.;
	#endif

	// evaluate the brdf
	float Kd = max(0., dot(N, L));
	float Ks = 0.;

	#if USE_SPOT_LIGHT
		vec3 spotDir = uLight.direction;
		att *= smoothstep(uLight.outerAngleCos, uLight.innerAngleCos, dot(-L, spotDir));
	#endif

	if (Kd > 0.)
	{
		vec3 V = normalize(-ex_PositionInView);
		vec3 H = normalize(V + L);

		Ks = max(0., pow(dot(N, H), uMaterial.shininess));

		#if NEED_LIGHT_SPACE
			vec3 shadowCoord = 0.5 + 0.5 * (ex_PositionInLight.xyz / ex_PositionInLight.w);

			#if USE_SHADOW_MAP
				// No PCF
				//shadowTerm = texture(shadowSampler, shadowCoord);

				// 3x3 PCF
				for (float shadowBiasV = -1.0; shadowBiasV <= 1.0; shadowBiasV += 1.0)
				for (float shadowBiasU = -1.0; shadowBiasU <= 1.0; shadowBiasU += 1.0)
					shadowTerm += texture(shadowSampler, vec3(shadowCoord.xy + vec2(shadowBiasU, shadowBiasV) * uRcpShadowSize, shadowCoord.z));
				shadowTerm /= 9.0;

				// 9x9 PCF
				//for (float shadowBiasV = -4.0; shadowBiasV <= 4.0; shadowBiasV += 1.0)
				//for (float shadowBiasU = -4.0; shadowBiasU <= 4.0; shadowBiasU += 1.0)
				//	shadowTerm += texture(shadowSampler, vec3(shadowCoord.xy + vec2(shadowBiasU, shadowBiasV) * uRcpShadowSize, shadowCoord.z));
				//shadowTerm /= 81.0;
			#endif
			#if USE_COOKIE_MAP
				#if USE_POINT_LIGHT
					// TODO
					//lightColor *= texture(cookieSampler, ).rgb;
				#else
					lightColor *= texture(cookieSampler, shadowCoord.xy).rgb;
				#endif
			#endif
		#endif
	}

	// compute the light's intensity
	vec3 diffuseIntensity = Kd * att * uLight.intensity * lightColor;
	vec3 specularIntensity = Ks * att * uLight.intensity * lightColor;

	// output the final color
	out_FragColor = vec4(shadowTerm * (diffuseIntensity * diffuseReflectance + specularIntensity * specularReflectance), uMaterial.opacity);
#endif
}
