#version 430

// these are only for the code that runs in single workgroup
#define GROUP_SIZE_X 8
#define GROUP_SIZE_Y 8
layout (local_size_x = GROUP_SIZE_X, local_size_y = GROUP_SIZE_Y) in;

#define MATERIAL_PROPERTIES_BINDING 1

#include <shaders/materials/commons_deferred.glsl>
#include <shaders/materials/commons.glsl>
#include <shaders/commons_hlsl.glsl>

//#define USE_AMBIENT_OCCLUSION_TERM

#define MRB_Composite                     0
#define MRB_Albedo                        1
#define MRB_Emissive                      2
#define MRB_Normals                       3
#define MRB_Roughness                     4
#define MRB_Lighting                      5
#define MRB_VoxelLighting                 6
#define MRB_Lighting_VoxelLighting        7
#define MRB_Diffuse_VoxelOcclusion        8
#define MRB_Specular_VoxelOcclusion       9
#define MRB_Diffuse_ScreenSpaceOcclusion 10
#define MRB_Diffuse_Occlusion_Combined   11
#define MRB_Material                     12
#define MRB_RTMask                       13
#define MRB_Raytrace                     14
#define MRB_Raytrace_Transparency        15
#define MRB_VolumetricFog                16
#define MRB_Particles                    17
#define MRB_ComponentTags                18
#define MRB_Debug                        19

uniform sampler2D  sCompositePrevious;
uniform sampler2D  sAlbedo;
uniform sampler2D  sEmissive;
uniform sampler2D  sTextureDepth;
uniform sampler2D  sTextureDepthPrevious;
uniform usampler2D sNormalMaterial;
uniform usampler2D sMetalnessRoughnessMaterialTags;
uniform sampler2D  sShadow;
uniform sampler2D  sScreenSpaceOcclusion;
uniform sampler2D  sSSR;
uniform sampler2D  sVoxelLighting;
uniform sampler2D  sVoxelOcclusion;		// RG8
uniform sampler2D  sWireframeColor;
uniform sampler2D  sWireframeDepth;
uniform sampler2D  sRaytrace;
uniform sampler2D  sVolumetricFog;
uniform sampler2D  sVolumetricLight;
uniform sampler2D  sTargetPrevious;
uniform sampler2D  sParticles;
uniform sampler2D  sDebug;

layout(rgba32f) uniform image2D imRaytraceHitDirectionPrimitiveIdReflection;
layout(rgba16f) uniform image2D imRaytraceShadedReflection;
layout(rgba32f) uniform image2D imRaytraceHitDirectionPrimitiveIdTransparency;
layout(rgba16f) uniform image2D imRaytraceShadedTransparency;

layout(rgba16f) uniform image2D imTarget;

uniform sampler2DArray s_BlueNoise;

struct CompositeParams
{
	float ssr_scaling_factor;
	int   composite_output_mode;
	float raytrace_scaling_factor;
	float _pad1;
};

layout (std140, row_major) uniform CompositeParamsBuffer
{
	CompositeParams composite_params;
};

layout (std140, row_major) uniform DeferredCompositeSetupBuffer
{
	DeferredCompositeSetup composite_setup;
};

layout (std140, row_major) uniform DeferredParams
{
	DispatchDeferredParams dispatch_setup;
};

vec3 TurboColormap(in float x)
{
  const vec4 kRedVec4 = vec4(0.13572138, 4.61539260, -42.66032258, 132.13108234);
  const vec4 kGreenVec4 = vec4(0.09140261, 2.19418839, 4.84296658, -14.18503333);
  const vec4 kBlueVec4 = vec4(0.10667330, 12.64194608, -60.58204836, 110.36276771);
  const vec2 kRedVec2 = vec2(-152.94239396, 59.28637943);
  const vec2 kGreenVec2 = vec2(4.27729857, 2.82956604);
  const vec2 kBlueVec2 = vec2(-89.90310912, 27.34824973);
  
  x = clamp(x, 0.0, 1.0);
  vec4 v4 = vec4( 1.0, x, x * x, x * x * x);
  vec2 v2 = v4.zw * v4.z;
  return vec3(
    dot(v4, kRedVec4)   + dot(v2, kRedVec2),
    dot(v4, kGreenVec4) + dot(v2, kGreenVec2),
    dot(v4, kBlueVec4)  + dot(v2, kBlueVec2)
  );
}

float linearizeDepth(float d)
{
	return dispatch_setup.near_far_plane.z / (dispatch_setup.near_far_plane.y + dispatch_setup.near_far_plane.x - d * dispatch_setup.near_far_plane.w);
}

vec3 worldPositionFromDepth(vec4 vDirection, float depth)
{
	return dispatch_setup.camera_position.xyz + vDirection.xyz * depth;
}

vec3 positionFromDepth(vec3 vDirection, float depth)
{
	return vDirection.xyz * depth;
}

vec3 positionFromDepthCurrent(vec3 vDirection, float depth)
{
	return (dispatch_setup.mat_model * vec4(vDirection.xyz * depth, 1.0)).xyz;
}

vec3 positionFromDepthPrevious(vec3 vDirection, float depth)
{
	return (dispatch_setup.mat_model_previous * vec4(vDirection.xyz * depth, 1.0)).xyz;
}

vec3 local_position_from_depth(vec3 vDirection, float depth)
{
	return vDirection.xyz * depth;
}

//layout(location = 0) out vec4 outColor;

// random functions: https://stackoverflow.com/questions/4200224/random-noise-functions-for-glsl
// A single iteration of Bob Jenkins' One-At-A-Time hashing algorithm.
uint hash( uint x ) {
	x += ( x << 10u );
	x ^= ( x >>  6u );
	x += ( x <<  3u );
	x ^= ( x >> 11u );
	x += ( x << 15u );
	return x;
}

// Compound versions of the hashing algorithm I whipped together.
uint hash( uvec2 v ) { return hash( v.x ^ hash(v.y)                         ); }
uint hash( uvec3 v ) { return hash( v.x ^ hash(v.y) ^ hash(v.z)             ); }
uint hash( uvec4 v ) { return hash( v.x ^ hash(v.y) ^ hash(v.z) ^ hash(v.w) ); }

// Construct a float with half-open range [0:1] using low 23 bits.
// All zeroes yields 0.0, all ones yields the next smallest representable value below 1.0.
float floatConstruct( uint m ) {
	const uint ieeeMantissa = 0x007FFFFFu; // binary32 mantissa bitmask
	const uint ieeeOne      = 0x3F800000u; // 1.0 in IEEE binary32

	m &= ieeeMantissa;                     // Keep only mantissa bits (fractional part)
	m |= ieeeOne;                          // Add fractional part to 1.0

	float  f = uintBitsToFloat( m );       // Range [1:2]
	return f - 1.0;                        // Range [0:1]
}

// Pseudo-random value in half-open range [0:1].
float random( float x ) { return floatConstruct(hash(floatBitsToUint(x))); }
float random( vec2  v ) { return floatConstruct(hash(floatBitsToUint(v))); }
float random( vec3  v ) { return floatConstruct(hash(floatBitsToUint(v))); }
float random( vec4  v ) { return floatConstruct(hash(floatBitsToUint(v))); }

#include <shaders/materials/commons.glsl>

#ifndef COMPOSITE_SIMPLE
vec4 sample_ssr(vec2 p, float depth)
{
	p *= composite_params.ssr_scaling_factor;

#if 1
#define R 2

	int x, y;
	float s = 0.0;

	vec4 v = vec4(0.0);
	float r = 10.5 * clamp(1.0 / depth, 0.5, 2.0);

	v = texture(sSSR, p);
	s = v.a;
	v.rgb *= s;

	for(y = -R; y <= R; y+=1)
	{
		for(x = -R; x <= R; x+=1)
		{
			if (x == 0 && y == 0)
				continue;

			if (s < 1.0)
			{
				ivec2 smpl_pos = ivec2(p * dispatch_setup.resolution) + ivec2(x, y);
		
				vec3 blue_noise = texelFetch(s_BlueNoise, ivec3(smpl_pos % ivec2(128), 0), 0).rgb;
				vec4 pxl = texture(sSSR, vec2(smpl_pos) * dispatch_setup.inv_resolution);

				if (pxl.a > s) // best match
				{
					v.rgb = pxl.rgb * pxl.a;
					s = pxl.a;
				}
			}
		}
	}

	return vec4(v.rgb, s);

#undef R
#else
	return texture(sSSR, p);
#endif

}

vec4 sample_rt(uvec2 pixel_pos, vec2 p, float depth, vec3 ref_normal)
{

	float ref_metalness;
	float ref_roughness;
	uint  ref_materialIndex;
	decode_metalness_roughness_material(texelFetch(sMetalnessRoughnessMaterialTags, ivec2(pixel_pos), 0).rg, ref_metalness, ref_roughness, ref_materialIndex);
	MaterialPropertiesGPU ref_material = materials.material_properties[ref_materialIndex];
	bool is_raytraced = (ref_material.flags & (MaterialFlag_Reflective)) != 0;

	if (is_raytraced == false)
		return vec4(0.0);

	const float KERNEL_5[5] = 
	{
		1.0/16.0, 1.0/4.0, 3.0/8.0, 1.0/4.0, 1.0/16.0
	};

	
	const float KERNEL_7[7] = 
	{
		0.07130343198685299,
		0.13151412084312236,
		0.1898792328888381,
		0.214606428562373,
		0.1898792328888381,
		0.13151412084312236,
		0.07130343198685299
	};

	const float KERNEL_11[11] = 
	{
		0.009300040045324049,
		0.028001560233780885,
		0.06598396774984912,
		0.12170274650962626,
		0.17571363439579307,
		0.19859610213125314,
		0.17571363439579307,
		0.12170274650962626,
		0.06598396774984912,
		0.028001560233780885,
		0.009300040045324049
	};

	p *= composite_params.raytrace_scaling_factor;
	uvec2 pixel_pos_scaled = pixel_pos / uvec2(1.0 / composite_params.raytrace_scaling_factor);

#if 1
#define R 3
//#define KERNEL KERNEL_5
#define KERNEL KERNEL_7
//#define KERNEL KERNEL_11

	vec4 v = vec4(0.0);
	float total_w = 0.0;

	for(int y = -R; y <= R; y+=1)
	{
		for(int x = -R; x <= R; x+=1)
		{
			ivec2 sampling_pos =  ivec2(pixel_pos) + ivec2(x, y);
			vec2 sampling_pos_scaled =  vec2(pixel_pos_scaled) + vec2(x, y) * composite_params.raytrace_scaling_factor + 0.5;

			float w = KERNEL[x + R] * KERNEL[y + R];

			{
				float metalness;
				float roughness;
				uint materialIndex;
				decode_metalness_roughness_material(texelFetch(sMetalnessRoughnessMaterialTags, sampling_pos, 0).rg, metalness, roughness, materialIndex);

				uint encoded_normal_material = texelFetch(sNormalMaterial, sampling_pos, 0).r;
				vec3 normal = decode_normal(encoded_normal_material);

				if (materialIndex != ref_materialIndex)
					w = 0.0;

				MaterialPropertiesGPU material = materials.material_properties[materialIndex];
				bool is_raytraced = (material.flags & (MaterialFlag_Reflective)) != 0;
				if (is_raytraced == false)
					w = 0.0;

				float normal_diff = clamp(dot(ref_normal, normal), 0.0, 1.0);
				w *= normal_diff;

				{
					//vec3 blue_noise = texelFetch(s_BlueNoise, ivec3(smpl_pos % ivec2(128), 0), 0).rgb;
					vec4 pxl = texture(sRaytrace, sampling_pos_scaled * dispatch_setup.inv_resolution);
					//vec4 pxl = texelFetch(sRaytrace, sampling_pos_scaled, 0);

					{
						v.rgb += pxl.rgb * w;
						total_w += w;
					}
				}
			}
		}
	}

	return vec4(v.rgb / total_w, 1.0);
	//return vec4(total_w, total_w, total_w, s);

#undef R
#undef KERNEL
#else
	return texelFetch(sRaytrace, ivec2(pixel_pos_scaled), 0);
#endif

}
#endif

// simple gaussian-like filter, depth aware
vec4 sample_volumetric_fog(uvec2 pixel_pos, float ref_depth, float ref_depth_diff_tolerance, uint ref_material_idx, float luminance_ceiling)
{
#define R 3
#define KERNEL KERNEL_7
	const float KERNEL_5[5] = 
	{
		1.0/16.0, 1.0/4.0, 3.0/8.0, 1.0/4.0, 1.0/16.0
	};

	const float KERNEL_7[7] = 
	{
		0.07130343198685299,
		0.13151412084312236,
		0.1898792328888381,
		0.214606428562373,
		0.1898792328888381,
		0.13151412084312236,
		0.07130343198685299
	};

	const float KERNEL_11[11] = 
	{
		0.009300040045324049,
		0.028001560233780885,
		0.06598396774984912,
		0.12170274650962626,
		0.17571363439579307,
		0.19859610213125314,
		0.17571363439579307,
		0.12170274650962626,
		0.06598396774984912,
		0.028001560233780885,
		0.009300040045324049
	};

	vec4 vc       = vec4(0.0);
	float total_w = KERNEL[R] * KERNEL[R] * 20.0;
	vec4 c0       = texelFetch(sVolumetricFog, ivec2(pixel_pos), 0);
	
	//return c0;

	float alpha_w = 1.0;//total_w;
	float alpha = c0.a * alpha_w;

	c0 *= total_w;

	const int R_S = 1;
	for(int y = -R * R_S; y <= R * R_S; y += 1)
	{
		for(int x = -R * R_S; x <= R * R_S; x += 1)
		{
			if (x == 0 && y == 0)
				continue;	// should optimize-out

			ivec2 sampling_pos = ivec2(pixel_pos) + ivec2(x, y);
			float w = KERNEL[x + R] * KERNEL[y + R];

			{
				MetalnessRoughnessMeterialTags metalness_roughness_material_tags;
				metalness_roughness_material_tags = decode_metalness_roughness_material_tags(texelFetch(sMetalnessRoughnessMaterialTags, sampling_pos, 0).rgba);

				if (ref_material_idx != metalness_roughness_material_tags.material_index)
					w = 0.0;

				float depth = texelFetch(sTextureDepth, sampling_pos, 0).r;

				if (abs(depth - ref_depth) > ref_depth_diff_tolerance) // not linearized
					w = 0.0;

				if (w > 0.0)
				{
					vec4 c = texelFetch(sVolumetricFog, sampling_pos, 0);
					vc += c * w;
					total_w += w;

					alpha += c.a * 1.0;
					alpha_w += 1.0;
				}
			}
		}
	}

	// weight by 
	if (alpha_w > 0.0)
	{
		alpha = alpha / alpha_w;
		//alpha = exp(-alpha)
		alpha = sqrt(alpha);
	}
	else
		alpha = 0.0;

	// weight samples by total coverage? was trying to make things nicer
	// for very bright fog areas, but it seems we are better 'tonemapping' the actual fog?
	alpha = 1.0;

#undef R
#undef KERNEL
#if 1
	//return vec4(total_w);
	vc += c0;
	vc = vc * (alpha / total_w);

	// hdr-tonemapping, kind of, to clamp very bright cells (add configurability for highest brightness)
	float peak_brightness = luminance_ceiling;
	vc.rgb /= peak_brightness;
	vc.rgb = peak_brightness * (vc.rgb / (vc.rgb + 1.0));
	return vc;

#else
	return c0;
#endif

}

vec4 calculate_lighting_fill(in vec3 pos, in vec3 normal, in vec3 light_pos, in vec3 cam_pos, in float NdotL)
{
	float d = NdotL;
	if (d < 0.0)
		d = 0.0;
	
	vec3 specular = vec3(0.0);
	if (d > 0.0)
		specular = pow(max(0.0, dot(reflect(-normalize(pos - light_pos), normalize(normal)), -normalize(cam_pos - pos))), 14.0) * composite_setup.fill_color.rrr;

	return vec4(vec3(d) * composite_setup.fill_color.xyz + specular, 1.0);
	//return vec4(vec3(d) * colorDiffuse.xyz, 1.0);
}

void main()
{
	uvec2 tile_pos = gl_WorkGroupID.xy;
	uvec2 pixel_pos = tile_pos.xy * gl_WorkGroupSize.xy + gl_LocalInvocationID.xy;
	uvec2 pixel_pos_inv_y = pixel_pos;
	pixel_pos_inv_y.y = uint(dispatch_setup.resolution.y) - pixel_pos_inv_y.y;

	// basic world pos reconstruction

	vec3 view_direction;
	view_direction.x = -dispatch_setup.camera_projection_params.z + dispatch_setup.camera_projection_params.x * pixel_pos.x * dispatch_setup.inv_resolution.x;
	view_direction.y = -dispatch_setup.camera_projection_params.w + dispatch_setup.camera_projection_params.y * pixel_pos.y * dispatch_setup.inv_resolution.y;
	view_direction.z = 1.0;

#ifdef SPIRV_VULKAN
	view_direction.y = -view_direction.y;
#endif

	// noise for dithering
	float dither = 0.0;//random(tex_coord);

	// other stuff

	vec4 color;
	uint encoded_normal_material = texelFetch(sNormalMaterial, ivec2(pixel_pos), 0).r;
	vec3 normal = decode_normal(encoded_normal_material);
	vec4 albedo;

	int materialId = decode_material(encoded_normal_material);
	MaterialPropertiesGPU material;
	material.flags = 0;
	
	bool is_background = false;
	float depth = 0.0;

	if ((materialId & MATERIAL_ID_MASK_ATTR) == ATTR_BACKGROUND)
	{
		is_background = true;

		// background + volumetrics
		albedo = texelFetch(sAlbedo, ivec2(pixel_pos), 0);
		color = albedo;

		//vec4 shadow = texelFetch(sShadow, ivec2(pixel_pos), 0);
		//shadow.a = pow(max(0.0, shadow.a), 0.6);

		//color.a = 1.0;
		//color.rgb += shadow.a;

		//color.r = .0;
	}

	//imageStore(imTarget, ivec2(pixel_pos.xy), color);
	//return;

	{
		// view space normal (from world). 

		vec3 local_normal              = vec3(0.0, 0.0, -1.0);
		vec4 shadow                    = vec4(1.0, 1.0, 1.0, 0.0);
		vec4 voxel_light               = vec4(0.0);
		float voxel_occlusion_diffuse  = 0.0;
		float voxel_occlusion_specular = 0.0;
		float voxel_occlusion_combined = 0.0;

		albedo = texelFetch(sAlbedo, ivec2(pixel_pos), 0);

		if (is_background == false)
		{
			local_normal.x = dot(normal.xyz, dispatch_setup.mat_model[0].xyz);
			local_normal.y = dot(normal.xyz, dispatch_setup.mat_model[1].xyz);
			local_normal.z = dot(normal.xyz, dispatch_setup.mat_model[2].xyz);

			depth                    = texelFetch(sTextureDepth, ivec2(pixel_pos), 0).r;
			shadow                   = texelFetch(sShadow, ivec2(pixel_pos), 0);
			voxel_light              = texelFetch(sVoxelLighting, ivec2(pixel_pos), 0);
			voxel_occlusion_combined = voxel_light.a;

			vec2 voxel_occlusion     = texelFetch(sVoxelOcclusion, ivec2(pixel_pos), 0).rg;
			voxel_occlusion_diffuse  = voxel_occlusion.r;
			voxel_occlusion_specular = voxel_occlusion.g;

			{
				vec4 color = vec4(0.0);
				color.rgb = abs(local_normal);
				//imageStore(imTarget, ivec2(pixel_pos.xy), color);
				//return;
			}
		}

		float linear_depth = linearizeDepth(depth);

		//float hbao = max(texelFetch(sScreenSpaceOcclusion, ivec2(pixel_pos), 0).r, 0.0) * 0.75 + 0.25;
		float hbao = max(texelFetch(sScreenSpaceOcclusion, ivec2(pixel_pos), 0).r, 0.0);
		if (composite_setup.sso_strength > 0.0)
		{
			hbao = pow(hbao, composite_setup.sso_strength);
		}
		//hbao = pow(hbao, 1.5);

		vec4 ssr = vec4(0.0); 
		vec4 particles = texelFetch(sParticles, ivec2(pixel_pos), 0);
		float ssr_factor = 0.0;

		vec3 world = positionFromDepthCurrent(view_direction, linear_depth);

		MetalnessRoughnessMeterialTags metalness_roughness_material_tags;
		metalness_roughness_material_tags = decode_metalness_roughness_material_tags(texelFetch(sMetalnessRoughnessMaterialTags, ivec2(pixel_pos), 0).rgba);
		uint materialIndex = metalness_roughness_material_tags.material_index;

		// load proper material
		if (is_background == false)
			material = materials.material_properties[materialIndex];

		{
			// TODO: Blend with global ilumination. This is absolutely not correct, but kind of gives ok results for now
			//shadow.rgb = max(shadow.rgb, voxel_light.rgb * composite_setup.global_illumination_strength);// * (1.0 - metalness);
			//shadow.rgb += voxel_light.rgb * composite_setup.global_illumination_strength;// * (1.0 - metalness);
			
			//shadow.rgb += albedo.rgb * voxel_light.rgb * composite_setup.global_illumination_strength;// * (1.0 - metalness);
			shadow.rgb += shadow.rgb * voxel_light.rgb * composite_setup.global_illumination_strength;// * (1.0 - metalness);
			shadow.rgb += voxel_light.rgb * composite_setup.global_illumination_base_strength;// * (1.0 - metalness);
		}

		// xv ambient occlussion
#ifdef USE_AMBIENT_OCCLUSION_TERM
		shadow.rgb *= (1.0 - voxel_occlusion_combined * composite_setup.occlusion_strength);
#endif

		#ifndef COMPOSITE_SIMPLE
		if ((material.flags & MaterialFlag_ScreenspaceReflect) != 0)
		{
			ssr = sample_ssr(vec2(pixel_pos) * dispatch_setup.inv_resolution.xy, depth);
			//ssr.rgb = texture(sSSR, vTexcoord0 * 0.5).rgb * 10.0;
			ssr_factor = 1.0;
		}
		#endif

		//shadow.a = 1.0 - exp(-max(0.0, shadow.a));
		//shadow.a = pow(max(0.0, shadow.a), 0.6);

		// add some fill light
		//vec3 fill_color = hbao * vGlobalFillColor.xyz * max(dot(local_normal.xyz, normalize(-view_coords)), 0.0);
		vec4 fill_color = calculate_lighting_fill(world.xyz, local_normal.xyz, vec3(0.0), vec3(0.0), dot(local_normal.xyz, normalize(-world.xyz)));

		shadow.rgb += fill_color.rgb;
#ifdef USE_AMBIENT_OCCLUSION_TERM
		shadow.rgb = shadow.rgb * hbao;// + dither * 0.01;
#endif

		color.a = 1.0;
		//color.rgb = (albedo.rgb * vec3(max(0.0, 1.0 - shadow.a))) * (shadow.rgb) + composite_setup.global_ambient_color.rgb;
		if (is_background == false)
		{
			color.rgb = (vec3(max(0.0, 1.0 - shadow.a))) * (shadow.rgb) + composite_setup.ambient_color.rgb;
		}

        // scale shadow.a (opacity for volumetrics) by distance to the face
        shadow.a = pow(shadow.a * linear_depth / 10.0, 2.0);  // fog extinction

		// experimental for emmisive materials
		//color.rgb = mix(color.rgb, albedo.rgb * (0.5 + shadow.rgb * 0.5), color.a);

		color.rgb += ssr.rgb * 4.0;// * (shadow.rgb + composite_setup.global_ambient_color.rgb);
		color.rgb += shadow.a;

		if (composite_params.composite_output_mode == MRB_Composite)
		{
#ifdef USE_AMBIENT_OCCLUSION_TERM
			color.rgb *= (1.0 - voxel_occlusion_combined * composite_setup.occlusion_strength * 0.55);
#endif

			float fog_density = linear_depth;
			if (composite_setup.fog_range > 0.0)
				fog_density = fog_density / composite_setup.fog_range;
			else
				fog_density = 0.0;

			float fog_height_density = (1.0 - clamp(abs(world.y) / composite_setup.fog_height, 0.0, 1.0)) * composite_setup.fog_height_density;
			
			fog_density = exp(fog_density + fog_height_density) - 1.0;
			float fog_density_margin = 0.01;
			fog_density = clamp(fog_density, fog_density_margin, 1.0 - fog_density_margin);
			//fog_density = fog_density * fog_density;

			vec3 fog_color = composite_setup.fog_color.rgb;

			//color.rgb = vec3(fog_density);
			color.rgb = mix(color.rgb, fog_color, fog_density);

			// volumetric fog
			{
				//vec4 volumetric_fog_color = texelFetch(sVolumetricFog, ivec2(pixel_pos), 0);
				vec4 volumetric_fog_color = sample_volumetric_fog(
					pixel_pos,
					depth,
					is_background ? 1000000.0 : 0.01,
					materialIndex,
					composite_setup.volumetric_fog_luminance_ceiling
				);
				volumetric_fog_color.rgb *= composite_setup.volumetric_fog_tint.rgb;

				//if (composite_setup.volumetric_fog_intensity > 0.0)
				//	volumetric_fog_color.rgb = pow(volumetric_fog_color.rgb, vec3(composite_setup.volumetric_fog_intensity));
				if (composite_setup.volumetric_fog_intensity > 0.0)
					volumetric_fog_color.rgb = volumetric_fog_color.rgb * vec3(composite_setup.volumetric_fog_intensity);

				//volumetric_fog_color.rgb = vec3(0.0);
				float volumetric_fog_blend = 0.0;
				if (composite_setup.volumetric_fog_strength > 0.0)
				{
					volumetric_fog_blend = volumetric_fog_color.a;
					volumetric_fog_blend = 1.0 - volumetric_fog_blend;
					if (composite_setup.volumetric_fog_blend_function == VFB_ColorViaDensity)
					{
						volumetric_fog_blend = exp(-volumetric_fog_blend / composite_setup.volumetric_fog_strength);
					}
					else // composite_setup.volumetric_fog_blend_function == VFB_InverseDensityBlend
					{
						volumetric_fog_blend = 1.0 - clamp((1.0 - volumetric_fog_blend) * composite_setup.volumetric_fog_strength, 0.0, volumetric_fog_blend);
					}

					//volumetric_fog_blend = clamp(volumetric_fog_blend * composite_setup.volumetric_fog_intensity, 0.0, 1.0);
				}

				color.rgb = mix(color.rgb, volumetric_fog_color.rgb, volumetric_fog_blend);
				//color.rgb = volumetric_fog_color.rgb;
				//color.rgb = vec3(volumetric_fog_blend);

				if (composite_setup.transparency_blend == TB_Add)
					color.rgb += particles.rgb  * (1.0 - volumetric_fog_blend);
				if (composite_setup.transparency_blend == TB_Blend)
				{
					color.rgb = mix(color.rgb, particles.rgb, 1.0 - particles.a);
					//color.rgb = vec3(1.0 - volumetric_fog_blend);
					//color.rgb = vec3(1.0 - particles.a);
				}

				//color.rgb = vec3(volumetric_fog_color.rgb);
				//color.rgb = vec3(volumetric_fog_blend);
			}

			// volumetric lights
			{
				vec4 volumetric_light = texelFetch(sVolumetricLight, ivec2(pixel_pos), 0);

				//volumetric_fog_color.rgb = vec3(0.0);
				float blend = clamp(pow(volumetric_light.a, 0.2), 0.0, 1.0);
				//blend = 1.0 - blend;
				color.rgb = mix(color.rgb, volumetric_light.rgb * composite_setup.volumetric_light_strength, blend);
				//color.rgb = vec3(blend);
			}

		}
		#ifndef COMPOSITE_SIMPLE
		else if (composite_params.composite_output_mode == MRB_Albedo)
		{
			color.rgb = albedo.rgb;
			//color.rgb = vec3(fract(depth * 131.141));
		}
		else if (composite_params.composite_output_mode == MRB_Emissive)
		{
			color.rgb = texelFetch(sEmissive, ivec2(pixel_pos), 0).rgb;
		}
		else if (composite_params.composite_output_mode == MRB_Normals)
		{
			//color.rgb = (length(normal.xyz) - 1.0).xxx * 10000.0;
			color.rgb = normal.xyz * 0.5 + 0.5;
		}
		else if (composite_params.composite_output_mode == MRB_Roughness)
		{
			color.rgb = vec3(metalness_roughness_material_tags.roughness);
		}
		else if (composite_params.composite_output_mode == MRB_Lighting)
		{
			color.rgb = texelFetch(sShadow, ivec2(pixel_pos), 0).rgb;
		}
		else if (composite_params.composite_output_mode == MRB_Lighting_VoxelLighting)
		{
			color.rgb = max(texelFetch(sShadow, ivec2(pixel_pos), 0).rgb, albedo.rgb * voxel_light.rgb * composite_setup.global_illumination_strength);
		}
		else if (composite_params.composite_output_mode == MRB_VoxelLighting)
		{
			//color.rgb = albedo.rgb * voxel_light.rgb * composite_setup.global_illumination_strength;
			color.rgb = voxel_light.rgb * composite_setup.global_illumination_strength;
			//if (voxel_occlusion_combined = 1.0)
			//	color.rgb = vec3(1.0 - voxel_occlusion_combined);
		}
		else if (composite_params.composite_output_mode == MRB_Diffuse_VoxelOcclusion)
		{
			float vx_ao = 1.0 - max(voxel_occlusion_diffuse, 0.0);
			vx_ao = 0.15 + vx_ao * 0.85;

			color.rgb = vec3(vx_ao);
		}
		else if (composite_params.composite_output_mode == MRB_Specular_VoxelOcclusion)
		{
			float ao = 1.0 - max(voxel_occlusion_specular, 0.0);
			color.rgb = vec3(ao);
		}
		else if (composite_params.composite_output_mode == MRB_Diffuse_Occlusion_Combined)
		{
			float ss_ao = max(texelFetch(sScreenSpaceOcclusion, ivec2(pixel_pos), 0).r, 0.0);
			float vx_ao = 1.0 - max(voxel_occlusion_diffuse, 0.0);
			vx_ao = 0.15 + vx_ao * 0.85;

			ss_ao = 1.0 - clamp((1.0 - ss_ao) * composite_setup.occlusion_strength, 0.0, 1.0);
			vx_ao = 1.0 - clamp((1.0 - vx_ao) * composite_setup.occlusion_strength, 0.0, 1.0);

			float ao = min(ss_ao, vx_ao);

			color.rgb = vec3(ao);
		}
		else if (composite_params.composite_output_mode == MRB_Diffuse_ScreenSpaceOcclusion)
		{
			float hbao = max(texelFetch(sScreenSpaceOcclusion, ivec2(pixel_pos), 0).r, 0.0);
			color.rgb = vec3(hbao);
		}
		else if (composite_params.composite_output_mode == MRB_Material)
		{
			color.rgb = vec3(materialIndex);
		}
		else if (composite_params.composite_output_mode == MRB_RTMask)
		{
			uint material_flags = material.flags;

			// also check material overrides. only include 4 attributes
			if ((metalness_roughness_material_tags.material_flag_overrides & MaterialFlag_OverrideFlags) != 0)
			{
				material_flags &= ~0xf;
				material_flags |= metalness_roughness_material_tags.material_flag_overrides & 0xf;
			}

			color.rgb = clamp(color.rgb, 0.0, 1.0) * 0.15;
			if ((material_flags & (MaterialFlag_Reflective)) != 0)
				color.r = 1.0;

			if ((material_flags & (MaterialFlag_Transparent)) != 0)
				color.g = 1.0;
		}
		else if (composite_params.composite_output_mode == MRB_Raytrace)
		{
#if 1
			color.rgb = texelFetch(sRaytrace, ivec2(pixel_pos * composite_params.raytrace_scaling_factor), 0).rgb;
#else
			color.rgb = sample_rt(
				pixel_pos,
				vec2(pixel_pos) * dispatch_setup.inv_resolution.xy, 0.0,
				normal
			).rgb;
#endif
		}
		else if (composite_params.composite_output_mode == MRB_Raytrace_Transparency)
		{
			color.rgb = texelFetch(sRaytrace, ivec2(pixel_pos * composite_params.raytrace_scaling_factor), 0).aaa;
		}
		else if (composite_params.composite_output_mode == MRB_VolumetricFog)
		{
			//vec4 f = texelFetch(sVolumetricFog, ivec2(pixel_pos), 0).rgba;
			vec4 f = sample_volumetric_fog(
				pixel_pos,
				depth,
				is_background ? 1000000.0 : 0.01,
				materialIndex,
				composite_setup.volumetric_fog_luminance_ceiling
			);

			float blend = exp(-f.a / composite_setup.volumetric_fog_strength);
			color.rgb = f.rgb;
			//color.rgb = f.rgb * (1.0 - blend) * 10.0;
			//color.rgb = vec3(1.0 - blend);
		}
		else if (composite_params.composite_output_mode == MRB_Particles)
		{
			color.rgb = abs(particles.rgb);
		}
		else if (composite_params.composite_output_mode == MRB_ComponentTags)
		{
			color.rgb = vec3(fract(float(metalness_roughness_material_tags.component_tags) / 16.0));
		}
		else if (composite_params.composite_output_mode == MRB_Debug)
		{
			color.rgb = texelFetch(sDebug, ivec2(pixel_pos), 0).rgb;
		}
		#endif
		//color.rgb = vec3(hbao);

		if (false)
		{
			vec4 hit_direction_primitiveID = imageLoad(imRaytraceHitDirectionPrimitiveIdReflection, ivec2(pixel_pos));
			uint hit_primitiveID = asuint(hit_direction_primitiveID.a);
			if (hit_primitiveID != -1)
			{
				color.rgb = vec3(TurboColormap(fract(float(hit_primitiveID) * 0.13453)));
			}
		}

		if (false)
		{
			vec4 hit_direction_primitiveID = imageLoad(imRaytraceHitDirectionPrimitiveIdTransparency, ivec2(pixel_pos));
			uint hit_primitiveID = asuint(hit_direction_primitiveID.a);
			if (hit_primitiveID != -1)
			{
				color.rgb = vec3(TurboColormap(fract(float(hit_primitiveID) * 0.13453)));
			}
			else
			{
				color.r = 0.0;
			}
		}

		if (false)
		{
			vec4 rt_shaded_reflection = imageLoad(imRaytraceShadedReflection, ivec2(pixel_pos));
			if (rt_shaded_reflection.a != 0.0)
			{
				color.rgb = rt_shaded_reflection.rgb;
			}
		}

		if (false)
		{
			vec4 rt_shaded_transparency = imageLoad(imRaytraceShadedTransparency, ivec2(pixel_pos));
			if (rt_shaded_transparency.a != 0.0)
			{
				color.rgb = rt_shaded_transparency.rgb;
			}
			else
			{
				//color.rgb = vec3(1.0);
			}
		}

		if (false)
		{
			vec4 rt_shaded_reflection = imageLoad(imRaytraceShadedReflection, ivec2(pixel_pos));
			vec4 rt_shaded_transparency = imageLoad(imRaytraceShadedTransparency, ivec2(pixel_pos));
			if (rt_shaded_transparency.a != 0.0)
			{
				color.rgb = rt_shaded_transparency.rgb;
				color.rgb += rt_shaded_reflection.rgb;
			}
			else
			{
				color.rgb = vec3(1.0);
			}
		}
	}


	// composite wireframe if required (TODO: make this separate pass?)
	#ifndef COMPOSITE_SIMPLE
	if (true)
	{
		int search_dist = 1;

		vec4  wc  = texelFetch(sWireframeColor, ivec2(pixel_pos), 0);
		float wd = texelFetch(sWireframeDepth, ivec2(pixel_pos), 0).r;

		for(int iy = -search_dist; iy < search_dist; iy++)
		{
			for(int ix = -search_dist; ix < search_dist; ix++)
			{
				if (ix == 0 && iy == 0)
					continue;

				float wd_neighborhood = texelFetch(sWireframeDepth, ivec2(pixel_pos) + ivec2(ix, iy), 0).r;
				if (wd_neighborhood < wd)
				{
					wd = wd_neighborhood;
					wc = texelFetch(sWireframeColor, ivec2(pixel_pos) + ivec2(ix, iy), 0);
				}
			}
		}

		if (wd < 1.0)
		{
			if (wd < depth || (wd >= depth && is_background)) // NOTE: This offset causes non-linear blending problem. Maybe this should be disabled
				color.rgb = mix(color.rgb, wc.rgb, 0.9 * wc.a);
			else if (wd < 1.0)
			{
				float t = float(globals.monotonic & 0xffff) * 1.1;
				float pattern = step(0.5, fract(t + (pixel_pos.x * dispatch_setup.inv_resolution.x + pixel_pos.y * dispatch_setup.inv_resolution.y) * 60.01));
				pattern = pattern * 0.4 + 0.6;
				color.rgb = mix(color.rgb, wc.rgb * pattern * 0.5 * vec3(1.0, 0.3, 0.3), 0.9 * wc.a);
			}
		}
	}
	#endif
	//

	//outColor =  color;
	//outColor = texture(sShadow, vTexcoord0);
	//outColor = texture(sAlbedo, vTexcoord0);
	//outColor = textureLod(sVoxelLighting, tex_coord, 0.0).aaaa;
	//outColor.rgb *= (1.0 - outColor.a);

	//float metalness;
	//float roughness;
	//uint material;
	//decode_metalness_roughness_material(textureLod(sMetalnessRoughnessMaterial, tex_coord, 0.0).r, metalness, roughness, material);
	//outColor = vec4(roughness);

	//color.r = 1.0;
	#if 0
	if (isnan(color.r) || isinf(color.r))
		color.r = 1.0;
	if (isnan(color.g) || isinf(color.g))
		color.g = 1.0;
	if (isnan(color.b) || isinf(color.b))
		color.b = 1.0;
	#endif
	imageStore(imTarget, ivec2(pixel_pos.xy), color);

}