
#version 450
#extension GL_EXT_debug_printf : enable

#define LIGHT_PROPERTIES_BINDING 1
#define MATERIAL_PROPERTIES_BINDING 1

#define USE_AMBIENT_OCCLUSION_TERM

uniform sampler2D   sDepth;
uniform usampler2D  sNormalMaterial;
uniform sampler2D   sAlbedo;
uniform usampler2D  sMetalnessRoughnessMaterialTags;	// metalness roughness material index
uniform samplerCube sEnviromentMap;
uniform sampler2D   sRaytrace;

uniform usampler2D  sRaytraceHitInfo;

// new RT hit buffer and shaded samples
uniform sampler2D                        sRaytraceReflection;
layout(rgba32f) uniform readonly image2D imHitDirectionPrimitiveID;

uniform sampler2D   sScreenSpaceOcclusion;
uniform sampler2D   sVoxelLighting;
uniform sampler2D   sVoxelOcclusion;

#include <shaders/materials/commons_deferred.glsl>
#include <shaders/materials/commons_gradient.glsl>
#include <shaders/materials/commons.glsl>

struct DeferredRenderLightsParams
{
	vec2  frustum_shift;		// TODO: Remove this. It is still here because of the non-standard way this code is invoked
	vec2  resolution;
	int   lights_num;
	float env_map_intensity;
	float raytrace_scaling_factor;
	float raytrace_strength;
};

layout(std140, row_major) uniform BasicDeferredParamsBuffer{
	BasicDeferredParams basic_params;
};

layout(std140, row_major) uniform DeferredRenderLightsParamsBuffer {
	DeferredRenderLightsParams render_lights_params;
};

layout (std140, row_major) uniform DeferredCompositeSetupBuffer {
	DeferredCompositeSetup composite_setup;
};

uniform sampler2D      s_BRDF;
uniform sampler2DArray s_BlueNoise;

in vec2 vTexcoord0;
in vec4 vFrustum;

// output

out vec4 outColor;

#include <shaders/commons_hlsl.glsl>
#include <shaders/materials/commons.glsl>
#include <shaders/deferred/lighting/lighting_support.glsl>

float linearizeDepth(float d)
{
	return basic_params.camera_near_far_plane.z / (basic_params.camera_near_far_plane.y + basic_params.camera_near_far_plane.x - d * basic_params.camera_near_far_plane.w);
}

vec3 positionFromDepth(vec3 vDirection, float depth)
{
	return vDirection.xyz * depth;
}


#include <shaders/materials/noise/noise3d.glsl>

float computeLODFromRoughness(float perceptual_roughness)
{
	return perceptual_roughness * 4.0;
}

struct IBLOutput
{	
	vec3 color_weighted;
	vec3 color_raw;
};

IBLOutput ibl(vec3 n, vec3 v, vec3 diffuseColor, vec3 specularColor, vec3 f0, vec3 f90, float perceptual_roughness)
{
	float NdotV = clamp(dot(n, v), 0.0, 1.0);
	vec3 r = reflect(-v, n);

	IBLOutput ibl;

#if 1
	float mip_count = 7.5;
	float lod = clamp(perceptual_roughness * mip_count, 0.0, float(mip_count));
	vec2 brdf_sample_point = clamp(float2(NdotV, 1.0 - perceptual_roughness), vec2(0.0, 0.0), vec2(1.0, 1.0));

	vec2 brdf = texture(s_BRDF, brdf_sample_point).rg;
	vec3 diffuseLight = textureLod(sEnviromentMap, n, 5.0).rgb;
	vec3 specularLight = textureLod(sEnviromentMap, r, lod).rgb * (brdf.x + brdf.y);

	//vec3 ibl = (specularColor * specularLight) * render_lights_params.env_map_intensity;
	//vec3 ibl = (diffuseColor * diffuseLight) * render_lights_params.env_map_intensity;

	// output two values, used for noise separation. diffuse and color are/can be noisy
	ibl.color_weighted = (diffuseColor * diffuseLight + specularColor * specularLight) * render_lights_params.env_map_intensity;
	ibl.color_raw      = (diffuseLight + specularLight) * render_lights_params.env_map_intensity;
	return ibl;
#else
	vec3 Ld = textureLod(sEnviromentMap, n, 6.0).rgb * diffuseColor; // sample "diffuse" LOD
	float lod = computeLODFromRoughness(perceptual_roughness);
	vec3 fake_spec = textureLod(sEnviromentMap, r, lod).rgb;
	return (Ld + fake_spec) * render_lights_params.env_map_intensity;
#endif
}

vec4 sample_rt(uvec2 pixel_pos, vec2 p, float rt_scaling_factor, float depth, vec3 ref_normal)
{
	float ref_metalness;
	float ref_roughness;
	uint  ref_materialIndex;
	decode_metalness_roughness_material(texelFetch(sMetalnessRoughnessMaterialTags, ivec2(pixel_pos), 0).rg, ref_metalness, ref_roughness, ref_materialIndex);
	MaterialPropertiesGPU ref_material = materials.material_properties[ref_materialIndex];
	bool is_raytraced = (ref_material.flags & MaterialFlag_Reflective) != 0;

	if (is_raytraced == false)
		return vec4(0.0);

	const float KERNEL_5[5] = 
	{
		1.0/16.0, 1.0/4.0, 3.0/8.0, 1.0/4.0, 1.0/16.0
	};

	const float KERNEL_7[7] = 
	{
		0.07130343198685299,
		0.13151412084312236,
		0.1898792328888381,
		0.214606428562373,
		0.1898792328888381,
		0.13151412084312236,
		0.07130343198685299
	};

	const float KERNEL_11[11] = 
	{
		0.009300040045324049,
		0.028001560233780885,
		0.06598396774984912,
		0.12170274650962626,
		0.17571363439579307,
		0.19859610213125314,
		0.17571363439579307,
		0.12170274650962626,
		0.06598396774984912,
		0.028001560233780885,
		0.009300040045324049
	};

	p *= rt_scaling_factor;
	uvec2 pixel_pos_scaled = pixel_pos / uvec2(1.0 / rt_scaling_factor);

#if 1
#define R 3
//#define KERNEL KERNEL_5
#define KERNEL KERNEL_7
//#define KERNEL KERNEL_11

	vec4 v = vec4(0.0);
	float total_w = 0.0;

	for(int y = -R; y <= R; y+=1)
	{
		for(int x = -R; x <= R; x+=1)
		{
			ivec2 sampling_pos =  ivec2(pixel_pos) + ivec2(x, y);
			vec2 sampling_pos_scaled =  vec2(pixel_pos_scaled) + vec2(x, y) * rt_scaling_factor + 0.5;

			float w = KERNEL[x + R] * KERNEL[y + R];
			{
				float metalness;
				float roughness;
				uint materialIndex;
				decode_metalness_roughness_material(texelFetch(sMetalnessRoughnessMaterialTags, sampling_pos, 0).rg, metalness, roughness, materialIndex);

				if (materialIndex != ref_materialIndex)
					w = 0.0;

				MaterialPropertiesGPU material = materials.material_properties[materialIndex];
				bool is_raytraced = (material.flags & MaterialFlag_Reflective) != 0;
				if (is_raytraced == false)
					w = 0.0;

				uint encoded_normal_material = texelFetch(sNormalMaterial, sampling_pos, 0).r;
				vec3 normal = decode_normal(encoded_normal_material);

				float normal_diff = clamp(dot(ref_normal, normal), 0.0, 1.0);
				w *= normal_diff;

				{
					vec4 pxl = texture(sRaytrace, sampling_pos_scaled / render_lights_params.resolution);
					//vec4 pxl = texelFetch(sRaytrace, sampling_pos_scaled, 0);

					v += pxl * w;
					total_w += w;
				}
			}
		}
	}

	return v / total_w;
	//return vec4(total_w, total_w, total_w, s);

#undef R
#else
	return texelFetch(sRaytrace, ivec2(pixel_pos_scaled), 0);
#endif

}

vec3 get_view_direction(vec2 screen_pos)
{
	vec2 vd_pos = screen_pos - render_lights_params.frustum_shift.xy * render_lights_params.resolution.xy * vec2(0.5, -0.5);
	vec3 view_direction;

	view_direction.x = -basic_params.camera_projection_params.z + basic_params.camera_projection_params.x * vd_pos.x / render_lights_params.resolution.x;
	view_direction.y = -basic_params.camera_projection_params.w + basic_params.camera_projection_params.y * vd_pos.y / render_lights_params.resolution.y;
	view_direction.z = 1.0;

	#ifdef SPIRV_VULKAN
	view_direction.y = -view_direction.y;
	#endif

	return view_direction;
}

void main() {
	ivec2 screen_pos = ivec2(gl_FragCoord.xy);

	vec4  base_color              = texelFetch(sAlbedo, screen_pos, 0);
	uint  encoded_normal_material = texelFetch(sNormalMaterial, screen_pos, 0).r;
	vec3  vNorm                   = decode_normal(encoded_normal_material);
	int   materialId              = decode_material(encoded_normal_material);

	float depth                   = linearizeDepth(texelFetch(sDepth, screen_pos, 0).r);
	vec3  view_direction          = get_view_direction(vec2(screen_pos));
	//vec3  world                   = basic_params.camera_position.xyz + positionFromDepth(vFrustum.xyz, depth);
	vec3  world                   = (basic_params.mModel * vec4(positionFromDepth(view_direction, depth), 1.0)).xyz;

	outColor = vec4(0.0);

	MetalnessRoughnessMeterialTags metalness_roughness_material_tags;
	metalness_roughness_material_tags = decode_metalness_roughness_material_tags(texelFetch(sMetalnessRoughnessMaterialTags, screen_pos, 0).rgba);

	float metalness = metalness_roughness_material_tags.metalness;
	float roughness = metalness_roughness_material_tags.roughness;
	uint material   = metalness_roughness_material_tags.material_index;

	uint material_flags = materials.material_properties[material].flags;
	// also check material overrides. only include 4 attributes
	if ((metalness_roughness_material_tags.material_flag_overrides & MaterialFlag_OverrideFlags) != 0)
	{
		material_flags &= ~0xf;
		material_flags |= metalness_roughness_material_tags.material_flag_overrides & 0xf;
	}

	vec3 specularColor;
	vec3 diffuseColor;

	if ((material_flags & MaterialFlag_DisableLighting) != 0)
	{
		outColor.a = 0.0;
		return;
	}

	// NOTE: because the model of PBR we use, and it allowing for 0...1 value for base color it doesn't play
	// well with our totally mixed up pipe. For RT we do a hacky job and use color and magnitude separately.
	// How this is going to play? No idea

	bool is_background = (materialId & MATERIAL_ID_MASK_ATTR) == ATTR_BACKGROUND;
	bool is_raytraced  = (material_flags & (MaterialFlag_Reflective)) != 0;
	bool is_particle   = (material_flags & (MaterialFlag_ParticleLighting)) != 0;

	{
		vec3 f0 = vec3(0.04, 0.04, 0.04);
		diffuseColor = base_color.rgb * (vec3(1.0, 1.0, 1.0) - f0) * (1.0 - metalness);
		specularColor = mix(f0, base_color.rgb, metalness);
	}

	// Roughness is authored as perceptual roughness; as is convention,
    // convert to material roughness by squaring the perceptual roughness [2].
	float alphaRoughness = roughness * roughness;

	vec3 specularEnvironmentR0 = specularColor.rgb;
    // Anything less than 2% is physically impossible and is instead considered to be shadowing. Compare to "Real-Time-Rendering" 4th editon on page 325.
    float reflectance = max(max(specularColor.r, specularColor.g), specularColor.b);
    vec3 specularEnvironmentR90 = vec3(1.0, 1.0, 1.0) * clamp(reflectance * 25.0, 0.0, 1.0);

	MaterialInfo materialInfo =
    {
        roughness,
        specularEnvironmentR0,
        alphaRoughness,
        diffuseColor,
        specularEnvironmentR90,
        specularColor
    };

	vec3 view = normalize(basic_params.camera_position.xyz - world.xyz);
	vec3 outLightColor = vec3(0.0);

	if (is_background == false)
	{
		// NOTE: We start with 'inverted' occlusion values
		float rt_specular_occlusion = 1.0;

		vec2 vx_diffuse_specular_occlusion = vec2(1.0) - texelFetch(sVoxelOcclusion, screen_pos, 0).rg;
		vx_diffuse_specular_occlusion = clamp(vx_diffuse_specular_occlusion, 0.0, 1.0);	// why this is needed i have no idea (TODO)
		float vx_specular_occlusion = vx_diffuse_specular_occlusion.g;

		if (is_raytraced)
		{
			bool evaluated = false;
			// from new RT
			{
				vec4 hit_direction_primitiveID = imageLoad(imHitDirectionPrimitiveID, ivec2(screen_pos));
				uint hit_primitiveID_candidate = asuint(hit_direction_primitiveID.a);

				if (hit_primitiveID_candidate != -1)
				{
					vec4 rt_reflection = texelFetch(sRaytraceReflection, screen_pos, 0);
					rt_specular_occlusion = rt_reflection.a;
					evaluated = true;
				}
			}

			if (evaluated == false)
			{
				vec4  raytraceColor = texelFetch(sRaytrace, screen_pos, 0).rgba;
				uvec4 raytraceHitAttribute = texelFetch(sRaytraceHitInfo, screen_pos, 0).rgba;
				uint bounces = raytraceHitAttribute.r;

				if (bounces >= 2)
					rt_specular_occlusion = raytraceColor.a;
			}
		}

		float specular_occlusion = min(rt_specular_occlusion, vx_specular_occlusion);
		float diffuse_occlusion  = 0.0;

#ifdef USE_AMBIENT_OCCLUSION_TERM
		float ss_ao = max(texelFetch(sScreenSpaceOcclusion, screen_pos, 0).r, 0.0);
		float vx_ao = vx_diffuse_specular_occlusion.r;
		//float vx_ao = max(1.0 - texelFetch(sVoxelLighting, screen_pos, 0).a, 0.0);

		if (composite_setup.sso_strength > 0.0)
		{
			ss_ao = pow(ss_ao, composite_setup.sso_strength);
			vx_ao = pow(vx_ao, composite_setup.sso_strength);
			ss_ao = 1.0 - clamp((1.0 - ss_ao) * composite_setup.occlusion_strength, 0.0, 1.0);
			vx_ao = 1.0 - clamp((1.0 - vx_ao) * composite_setup.occlusion_strength, 0.0, 1.0);

			specular_occlusion = pow(specular_occlusion, composite_setup.sso_strength);
			specular_occlusion = 1.0 - clamp((1.0 - specular_occlusion) * composite_setup.occlusion_strength, 0.0, 1.0);
		}

		float ao = min(ss_ao, vx_ao);
		//float ao = ss_ao;
		//float ao = vx_ao;

		// in case of missing info about specular occlusion, allow faking it
		specular_occlusion = 1.0 - max(1.0 - specular_occlusion, (1.0 - ao) * composite_setup.occlusion_specular_from_diffuse);

		diffuse_occlusion  = 1.0 - ao;
		specular_occlusion = 1.0 - specular_occlusion;

#endif
		IBLOutput ibl = ibl(vNorm, view, materialInfo.diffuseColor.rgb * (1.0 - diffuse_occlusion), materialInfo.specularColor.rgb * (1.0 - specular_occlusion), specularEnvironmentR0, specularEnvironmentR90, roughness);
		//vec3 ibl_color = ibl(vNorm, view, materialInfo.diffuseColor.rgb, materialInfo.specularColor.rgb, specularEnvironmentR0, specularEnvironmentR90, materials.material_properties[material].roughness);
		//ibl_color *= (1.0 - diffuse_occlusion);

		outColor.rgb = ibl.color_weighted;
		// NOTE: Don't output combined diffuse and specular occlusion. This is used to affect
		// lighting and RT, but if anything, we should have it split!
		// Specular should be used for mirror reflections (RT)

		//outColor.a   = saturate(diffuse_occlusion + specular_occlusion);
		outColor.a   = saturate(diffuse_occlusion);
		
		//outColor.rgb = vec3(specular_occlusion);
		//outColor.rgb = materialInfo.diffuseColor.rgb;// * (1.0 - diffuse_occlusion);
		//outColor.rgb = materialInfo.specularColor.rgb * (1.0 - specular_occlusion);
		//outColor.rgb = vec3(diffuse_occlusion);

		//outColor.rgb = vec3(rt_specular_occlusion);
	}

}
