#version 460

#define LIGHT_PROPERTIES_BINDING 1
#ifndef MATERIAL_PROPERTIES_BINDING
#define MATERIAL_PROPERTIES_BINDING 2
#endif

#include <shaders/materials/commons.glsl>
#include <shaders/materials/commons_rand.glsl>
#include <shaders/commons_hlsl.glsl>
#include <shaders/materials/noise/noise3d.glsl>
#include <shaders/particles/particles_commons.glsl>

uniform sampler2DArray s_BlueNoise;
uniform sampler2D s_NoiseRGBA;

#include <shaders/deferred/lighting/lighting_support.glsl>

layout(std140, row_major) uniform TransformParamsBuffer{
	EntityTransformParams transform_params;
};

layout(std430) buffer readonly restrict RibbonPositionsData
{
	float position[];
} prt_position_snapshots;

struct ParticleRibbonState
{
	float generation_id;
	uint segment_info;
};

layout(std430) buffer readonly restrict RibbonStatesData
{
	ParticleRibbonState states[];
} prt_states_snapshots;

struct RibbonRenderParams
{
	int   max_segments;
	float voxelize_thickness;
	int   base_segment;
	int   particles_per_segment;
	float thickness;
	int   lights_num;
	int   material_index;
	float first_segment_time_factor;
	int   render_as_strips;
	int   segment_subdivisions;
	int   _pad1;
	int   _pad2;
};

layout(std140) uniform RibbonRenderParamsBuffer {
	RibbonRenderParams ribbon_render_params;
};

struct ParticleRibbonSegmentInfo
{
	uint segments; // 0...max_segments-1
	uint first; // first is always in range, rolling!
	uint previous; // not stored, just for convenience
};

ParticleRibbonSegmentInfo ribbon_decode_segment_info(uint segment_info)
{
	ParticleRibbonSegmentInfo si;
	si.segments = segment_info & 0xffff;
	si.first = segment_info >> 16;
	si.previous = si.first - 1;
	if (si.first == 0)
		si.previous = ribbon_render_params.max_segments - 1;
	return si;
}

layout(location = 1) out struct
{
	vec3    vLocalPos;
	f16vec4 vColor;
	f16vec2 vUV0;
	uint    vIdx;
} vtx_output;
layout(location = 0) out uint instanceID;


int calculate_vidx_for_ribbon_segment(uint ribbon_id, int ribbon_segment)
{
	ParticleRibbonSegmentInfo ribbon_si = ribbon_decode_segment_info(prt_states_snapshots.states[ribbon_id].segment_info);
	
	ribbon_segment = int(ribbon_si.first) - ribbon_segment;
	if (ribbon_segment < 0)
		ribbon_segment += ribbon_render_params.max_segments;

	int vidx = ribbon_segment * ribbon_render_params.particles_per_segment + int(ribbon_id);
	return vidx;
}

uint get_ribbon_segments(uint ribbon_id)
{
	ParticleRibbonSegmentInfo ribbon_si = ribbon_decode_segment_info(prt_states_snapshots.states[ribbon_id].segment_info);
	int s = max(0, min(ribbon_render_params.max_segments, int(ribbon_si.segments)) - 1);
	return max(1, s);
}

vec3 ribbon_get_coords_for_vidx(int vidx)
{
	return vec3(prt_position_snapshots.position[vidx * 3 + 0], prt_position_snapshots.position[vidx * 3 + 1], prt_position_snapshots.position[vidx * 3 + 2]);
}

vec3 ribbon_get_coords_for_segment(uint ribbon_id, int segment_idx)
{
	uint vidx = calculate_vidx_for_ribbon_segment(ribbon_id, segment_idx);
	return vec3(prt_position_snapshots.position[vidx * 3 + 0], prt_position_snapshots.position[vidx * 3 + 1], prt_position_snapshots.position[vidx * 3 + 2]);
}

vec3 ribbon_get_coords_for_t(uint ribbon_idx, uint segment_idx, float t)
{
	int control_points_num = int(get_ribbon_segments(ribbon_idx)) + 1;
	int s0 = int(segment_idx);
	int s1 = min(s0 + 1, control_points_num - 1); // might be same as s0

	int vidx0 = calculate_vidx_for_ribbon_segment(ribbon_idx, s0);
	int vidx1 = calculate_vidx_for_ribbon_segment(ribbon_idx, s1);
	vec3 p0 = ribbon_get_coords_for_vidx(vidx0);
	vec3 p1 = ribbon_get_coords_for_vidx(vidx1);

	return mix(p0, p1, t);
}

vec3 catmull_rom(vec3 p0, vec3 p1, vec3 p2, vec3 p3, float t)
{
	float t2 = t * t;
	float t3 = t2 * t;

	return 0.5 * (
		(2.0 * p1) +
		(-p0 + p2) * t +
		(2.0 * p0 - 5.0 * p1 + 4.0 * p2 - p3) * t2 +
		(-p0 + 3.0 * p1 - 3.0 * p2 + p3) * t3
	);
}

vec3 b_spline(vec3 p0, vec3 p1, vec3 p2, vec3 p3, float t)
{
	float t2 = t * t;
	float t3 = t2 * t;

	vec4 T = vec4(t3, t2, t, 1.0);

	mat4 B = (1.0 / 6.0) * mat4(
        -1.0, 3.0, -3.0, 1.0,
         3.0, -6.0, 0.0, 4.0,
        -3.0, 3.0, 3.0, 1.0,
         1.0, 0.0, 0.0, 0.0
    );

	vec4 coeffs = T * B;

	return coeffs.x * p0 +
           coeffs.y * p1 +
           coeffs.z * p2 +
           coeffs.w * p3;
}

vec3 ribbon_get_coords_for_t_spline(uint ribbon_idx, uint segment_idx, float t)
{
	//return ribbon_get_coords_for_t(ribbon_idx, segment_idx, t);

	int control_points_num = int(get_ribbon_segments(ribbon_idx)) + 1;
	int s1 = int(segment_idx);
	int s2 = min(s1 + 1, control_points_num - 1); // might be same as s0

	int s0 = max(s1 - 1, 0);
	int s3 = min(s2 + 1, control_points_num - 1);

	int vidx0 = calculate_vidx_for_ribbon_segment(ribbon_idx, s0);
	int vidx1 = calculate_vidx_for_ribbon_segment(ribbon_idx, s1);
	int vidx2 = calculate_vidx_for_ribbon_segment(ribbon_idx, s2);
	int vidx3 = calculate_vidx_for_ribbon_segment(ribbon_idx, s3);
	vec3 p0 = ribbon_get_coords_for_vidx(vidx0);
	vec3 p1 = ribbon_get_coords_for_vidx(vidx1);
	vec3 p2 = ribbon_get_coords_for_vidx(vidx2);
	vec3 p3 = ribbon_get_coords_for_vidx(vidx3);

	if (s0 == s1)	// no previous point
		p0 = p1 - (p2 - p1);
	if (s3 == s2)	// no next after second point
		p3 = p2 + (p2 - p1);

	// this is ugly attempt to 'normalize' the spline
	if (false)
	{
		float d = length(p1 - p0);
		p0 = p1 - normalize(p1 - p0) * d;
		p3 = p2 + normalize(p3 - p2) * d;
	}

	return catmull_rom(p0, p1, p2, p3, t);
	//return b_spline(p0, p1, p2, p3, t);
	
	vec3 a0 = p3 - p2 - p0 + p1;
	vec3 a1 = p0 - p1 - a0;
	vec3 a2 = p2 - p0;
	vec3 a3 = p1;

	float t2 = t * t;
	return a0 * t * t2 + a1 * t2 + a2 * t + a3;
}

float compute_t_from_subsegment(int sub_segment_idx)
{
	return float(sub_segment_idx) / float(ribbon_render_params.segment_subdivisions);
}

vec3 ribbon_get_coords_for_segment_t(uint ribbon_id, int segment_idx, float t)
{
	return ribbon_get_coords_for_t_spline(ribbon_id, segment_idx, t);
}

// Custom vertex modifiers params

#ifdef INLINE_VERTEX_MODIFIERS_PARAMS_BLOCK

struct VertexInput
{
	uint id;
	vec3 pos;
	vec3 norm;
	vec4 color;
	vec2 uv0;
};

#inline <INLINE_VERTEX_MODIFIERS_PARAMS_BLOCK>
#endif

#ifdef INLINE_VERTEX_MODIFIERS_TRANSFORM_LOCAL_BLOCK
vec3 apply_vtx_modifier(vec3 pos, vec3 norm, in out vec3 color, uint ribbon_id)
{
	VertexInput vtx_input;
	vtx_input.id    = ribbon_id;
	vtx_input.pos   = pos;
	vtx_input.norm  = norm;
	vtx_input.color = vec4(color, 1.0);
	vtx_input.uv0   = vec2(0.0, 0.0);

	ModifierFactor modifier_factor = modifier_factor_defaults();
	modifier_factor.factor      = 1.0;
	modifier_factor.hash        = uint(ribbon_id);
	modifier_factor.id          = uint(ribbon_id);
	modifier_factor.instance_id = uint(ribbon_id);

	modifier_factor.position    = pos;
	modifier_factor.is_spawned  = false;

	CoordinateSystemTrasforms cs_transforms;
	cs_transforms.mat_local_to_model     = transform_params.mModel;
	cs_transforms.mat_local_to_instance  = mat_identity();
	cs_transforms.mat_local_to_model_inv = transform_params.mModelInv;

#inline <INLINE_VERTEX_MODIFIERS_TRANSFORM_LOCAL_BLOCK>

	pos   = vtx_input.pos;
	color = vtx_input.color.rgb;
	return pos;
}
#else
vec3 apply_vtx_modifier(vec3 pos, vec3 norm, in out vec3 color, uint ribbon_id)
{
	return pos;
}
#endif

vec3 get_ribbon_transformed_segment_position(int ribbon_id, int segment_idx)
{
	vec3 p = ribbon_get_coords_for_segment(ribbon_id, segment_idx);

	// TODO: Apply modifiers here!

	vec3 dummy_normal = vec3(0.0);
	vec3 dummy_color = vec3(0.0);
	vec3 modified_p = apply_vtx_modifier(p, dummy_normal, dummy_color, ribbon_id);

	return vector_transform_by_mat43(modified_p, transform_params.mModelView);
}

vec3 get_ribbon_transformed_segment_position_t(int ribbon_id, int segment_idx, float t)
{
	vec3 p = ribbon_get_coords_for_segment_t(ribbon_id, segment_idx, t);

	// TODO: Apply modifiers here!

	vec3 dummy_normal = vec3(0.0);
	vec3 dummy_color = vec3(0.0);
	vec3 modified_p = apply_vtx_modifier(p, dummy_normal, dummy_color, ribbon_id);

	return vector_transform_by_mat43(modified_p, transform_params.mModelView);
}

void main()
{

	instanceID = SYS_InstanceIndex;

	// fetch input data
	// when subdivision is enabled, SYS_VertexIndex extends up to the multiply of original ribbon vertices
	// when segments == n, with subdivision it is: (segments - 1) * subdivisions + 1
	
	// When doing subdivisions, last vtx will only get ribbon_sub_segment == 0, nothing else.
	// But it is important to take this case into accoint (when fetching +1 vtx) that is is not NaN or something
	// If it is anything else we can lerp between it and it will cancel out
	
	int ribbon_id;
	int ribbon_segment;
	int ribbon_sub_segment; // 0... ribbon_render_params.segment_subdivisions - 1, so the fractionally it is [0...n/(n-1)]

	{
		ribbon_id = SYS_InstanceIndex;
		int n = int(SYS_VertexIndex);	// renderer as line strips
		
		ribbon_segment = n / ribbon_render_params.segment_subdivisions;
		ribbon_sub_segment = n - ribbon_segment * ribbon_render_params.segment_subdivisions;
	}

	uint ribbon_segments = get_ribbon_segments(ribbon_id);
	
	vtx_output.vLocalPos = vec3(0.0, 0.0, 0.0);
	vtx_output.vIdx = 0;
	
	// NOTE: Here it somehow prohibts shader optimization? Or just the vs <> fs ratio is so much skewed?:(
	// NOTE: With latest drivers seems to be faster again....
	if (!prt_is_alive(ribbon_id) || ribbon_segment > ribbon_segments || (ribbon_segment == ribbon_segments && ribbon_sub_segment > 0) || ribbon_segments <= 1)
	{
		gl_Position = vec4(0.0 / 0.0, 0.0 / 0.0, 0.0 / 0.0, 0.0 / 0.0);
		return;
	}

	//vec3 vInstPosition = ribbon_get_coords_for_segment(ribbon_id, ribbon_segment);
	vec3 vInstPosition = ribbon_get_coords_for_segment_t(ribbon_id, ribbon_segment, compute_t_from_subsegment(ribbon_sub_segment));
	
	vec3 vInstNormal = vec3(1.0, 0.0, 0.0);
	vec4 vInstColor = prt_get_color(ribbon_id);

	vInstPosition = apply_vtx_modifier(vInstPosition, vInstNormal, vInstColor.rgb, ribbon_id);

	ParticleState vState = prt_get_state(ribbon_id);

	bool is_last = ribbon_segment == ribbon_segments && ribbon_sub_segment == 0;
	bool is_last_section = ribbon_segment >= ribbon_segments - 1;
	bool is_first_section = ribbon_segment == 0;
	bool is_one_before_last = ribbon_segment == (ribbon_segments - 1) && ribbon_sub_segment == (ribbon_render_params.segment_subdivisions - 1);
	bool is_first = ribbon_segment == 0 && ribbon_sub_segment == 0;
	bool is_full = ribbon_segments == ribbon_render_params.max_segments - 1;

	if (!prt_is_alive(ribbon_id))
	{
		gl_Position = vec4(0.0 / 0.0, 0.0 / 0.0, 0.0 / 0.0, -1.0);
		return;
	}

	//
	// NOTE: Export main line coordinates as world coords. In addition export interpolant for the left-to-right side of the strip
	// so we can try to reconstruct proper 'tube' in ps.
	//

	vec3 pos = vInstPosition;

	vtx_output.vLocalPos = pos;

	vec3 vPos1 = pos;
	vec3 vPos = vector_transform_by_mat43(vPos1, transform_params.mModelView);

	MaterialPropertiesGPU material = materials.material_properties[ribbon_render_params.material_index];

	// adjust extrusion by remaining life (bit smoother disapearing)
	float remaining_lifetime_based_extrusion_modifier = 1.0;
	float remaining_lifetime_margin = 0.2;
	if (vState.life_time > vState.life_span - remaining_lifetime_margin)
	{
		remaining_lifetime_based_extrusion_modifier = vState.life_time - (vState.life_span - remaining_lifetime_margin);
		remaining_lifetime_based_extrusion_modifier /= (vState.life_span - remaining_lifetime_margin);
		remaining_lifetime_based_extrusion_modifier = 1.0 - remaining_lifetime_based_extrusion_modifier;
	}
	else if (vState.life_time < remaining_lifetime_margin)
	{
		remaining_lifetime_based_extrusion_modifier = vState.life_time;
		remaining_lifetime_based_extrusion_modifier /= remaining_lifetime_margin;
		remaining_lifetime_based_extrusion_modifier = 1.0 - remaining_lifetime_based_extrusion_modifier;
	}
	
	remaining_lifetime_based_extrusion_modifier = clamp(remaining_lifetime_based_extrusion_modifier, 0.0, 1.0);
	
	vtx_output.vColor = f16vec4(vInstColor);
	vtx_output.vColor.a = remaining_lifetime_based_extrusion_modifier;	// hijack
	vtx_output.vIdx = 1;
	
	// TODO: Can simplify this too for ortho
	gl_Position = vector_transform_by_mat_projection(vPos, transform_params.mProjection);
}


